(source you uploaded: :contentReference[oaicite:0]{index=0})

---

### `solve_casimir.py` (corrected: comments/docstring reference **F S + S F**; variable `R` renamed to `F`)
```python
#!/usr/bin/env python3
"""
solve_casimir.py

Toy-Model Casimir Construction (Volume 1 v1)

Ansatz:
  Q = A (F S + S F) + sum_{a=1..4} B_a O_a^2

In this toy normalization we set A = 0, and solve for B1..B4 so that
  Q == diag(+1, -1, +1, -1).
"""

import sympy as sp
import csv
from pathlib import Path

# 1) Toy-model tick operators (exactly as in Vol 1)
#    Basis order: |0,1>, |0,2>, |1,1>, |1,2>
#    F = Renewal (outward flip), S = Sink (inward flip)
F = sp.Matrix([
    [0, 0, 0, 0],
    [0, 0, 0, 0],
    [1, 0, 0, 0],
    [0, 1, 0, 0]
])
S = sp.Matrix([
    [0, 0, 1, 0],
    [0, 0, 0, 1],
    [0, 0, 0, 0],
    [0, 0, 0, 0]
])

# 2) Four standard basis-projectors
O1 = sp.diag(1, 0, 0, 0)
O2 = sp.diag(0, 1, 0, 0)
O3 = sp.diag(0, 0, 1, 0)
O4 = sp.diag(0, 0, 0, 1)

# 3) Normalization: set A = 0
A_val = 0

# 4) Symbols for B1…B4 only
B1, B2, B3, B4 = sp.symbols('B1 B2 B3 B4')

# 5) Build Q = Σ B_a O_a (note: O_a are projectors, so O_a^2 = O_a)
Q = B1*O1 + B2*O2 + B3*O3 + B4*O4  # since A_val*(F*S + S*F) = 0

# 6) Target diag(+1,-1,+1,-1)
TARGET = sp.diag(+1, -1, +1, -1)

# 7) Equations: Q[i,j] == TARGET[i,j] for all i,j
eqs = [sp.Eq(Q[i, j], TARGET[i, j]) for i in range(4) for j in range(4)]

# 8) Solve for B1…B4
sol = sp.solve(eqs, [B1, B2, B3, B4], dict=True)
if not sol:
    raise RuntimeError("No solution found — check your definitions.")
coeffs = sol[0]

# 9) Write to CSV
out_dir = Path('results')
out_dir.mkdir(exist_ok=True)

with open(out_dir / 'casimir_coeffs.csv', 'w', newline='') as f:
    writer = csv.writer(f)
    writer.writerow(['coeff', 'value'])
    for sym in (B1, B2, B3, B4):
        writer.writerow([str(sym), float(coeffs[sym])])

print("✔ Casimir coefficients written to results/casimir_coeffs.csv")
print("Solution:", coeffs)
